const campaignStates = {};

const statusNames = {
    1: "Draft",
    2: "Moderation pending",
    3: "Rejected",
    4: "Ready",
    6: "Working",
    7: "Paused",
    8: "Stopped",
    9: "Completed"
};
/*
 * JavaScript for the standalone RASTY web panel.
 * Implements theme toggle, login/logout, campaign scheduling with per‑campaign run/pause
 * durations, global cycle control, and mode selection (normal, smart, advanced).
 */

// Utility for logging to the log section
// This function prints messages to the page log with colour‑coding based on the
// supplied type. Additionally, any log entry containing the word "stopped"
// (case‑insensitive) will be forced to appear in red to satisfy compliance
// requirements.
const logBox = document.getElementById('log');
function log(message, type = 'info') {
    const div = document.createElement('div');
    let colour;
    // Force stopped messages to display in red regardless of provided type
    const lowerMsg = String(message).toLowerCase();
    if (lowerMsg.includes('stopped')) {
        colour = '#f55';
    } else {
        switch (type) {
            case 'success': colour = '#0f0'; break;
            case 'warning': colour = '#ffa500'; break;
            case 'error':   colour = '#f55'; break;
            default:        colour = '#0ff';
        }
    }
    div.style.color = colour;
    const ts = new Date().toLocaleTimeString();
    div.textContent = `[${ts}] ${message}`;
    logBox.appendChild(div);
    logBox.scrollTop = logBox.scrollHeight;
    // Decide whether to trigger notifications based on user preferences and message content.
    // We treat "compliance" and generic errors (excluding specific status events) as always noteworthy,
    // but allow the user to toggle notifications for rejected campaigns and scheduler stops.
    let shouldNotify = false;
    const isRejection  = lowerMsg.includes('rejected');
    const isScheduler  = lowerMsg.includes('scheduler stopped');
    const isCompliance = lowerMsg.includes('compliance');
    const isCompleted  = lowerMsg.includes('completed');
    // Fetch event toggle states. If the elements are not found (e.g. on login page), defaults to false.
    const rejEl       = document.getElementById('notifyRejected');
    const schedulerEl = document.getElementById('notifySchedulerStopped');
    const completedEl = document.getElementById('notifyCompleted');
    const notifyRejected  = rejEl ? rejEl.checked : false;
    const notifyScheduler = schedulerEl ? schedulerEl.checked : false;
    const notifyCompleted = completedEl ? completedEl.checked : false;
    // Always notify on compliance messages.
    if (isCompliance) {
        shouldNotify = true;
    } else if (type === 'error' && !isRejection && !isScheduler && !isCompleted) {
        // Notify on generic errors that are not specific rejection, scheduler or completed events.
        shouldNotify = true;
    }
    // Conditionally notify on rejection, scheduler and completed events based on user toggles.
    if (isRejection && notifyRejected) {
        shouldNotify = true;
    }
    if (isScheduler && notifyScheduler) {
        shouldNotify = true;
    }
    if (isCompleted && notifyCompleted) {
        shouldNotify = true;
    }
    if (shouldNotify && typeof notifyAll === 'function') {
        notifyAll(message, type);
    }
}

/**
 * Update the visual API status indicator. When active is true the dot
 * becomes green and the text reads "API Active"; otherwise it becomes
 * red and reads "API Inactive". If the elements are not present this
 * function does nothing.
 *
 * @param {boolean} active Whether the API is active (statistics fetched)
 */
function updateApiStatus(active) {
    const dot = document.getElementById('apiStatusDot');
    const text = document.getElementById('apiStatusText');
    if (!dot || !text) return;
    if (active) {
        dot.classList.remove('api-inactive');
        dot.classList.add('api-active');
        text.textContent = 'API Active';
    } else {
        dot.classList.remove('api-active');
        dot.classList.add('api-inactive');
        text.textContent = 'API Inactive';
    }
}

/**
 * Query the PropellerAds API to determine whether a campaign is approved and
 * compliant. A campaign will only be considered approved if all creatives
 * associated with it have a policy_status of 1. If the API call fails
 * (network error or unexpected response) the function returns true to avoid
 * unnecessarily blocking campaign execution. You can modify the logic here
 * as more information about the API becomes available.
 *
 * @param {string|number} id    The numeric campaign ID
 * @param {string} token        The bearer token for API authentication
 * @returns {Promise<boolean>}  True if the campaign is approved
 */
async function isCampaignApproved(id, token) {
    try {
        const details = await callApi('GET', `/adv/campaigns/${id}`, token);
        // Some API responses nest creatives under a 'creatives' property. If no
        // creatives exist we assume approval to avoid false negatives.
        if (!details || !Array.isArray(details.creatives) || details.creatives.length === 0) {
            return true;
        }
        // Policy status of 1 typically means approved. Any other value
        // indicates pending or rejected status and should prevent playing.
        for (const creative of details.creatives) {
            if (typeof creative.policy_status !== 'number' || creative.policy_status !== 1) {
                // Mark as inactive and record a descriptive reason when a creative is not approved.
                if (!campaignStates[id]) campaignStates[id] = {};
                campaignStates[id].active = false;
                campaignStates[id].skipReason = 'Creative not approved';
                updateCampaignRowUI(id);
                return false;
            }
        }
        return true;
    } catch (err) {
        // If we cannot fetch status, err on the side of running and log a warning
        log(`Compliance check failed for campaign ${id}: ${err.message}`, 'warning');
        return true;
    }
}

// ----------------------- Notification helpers ----------------------------
/**
 * Send an email notification via the PHP backend. The backend expects a JSON
 * payload with keys: to, subject, text. If the call fails this function
 * silently ignores errors.
 * @param {string} to      Destination email address
 * @param {string} subject Email subject
 * @param {string} text    Email body
 */
async function sendEmailNotification(to, subject, text) {
    if (!to) return;
    try {
        await fetch('send_email.php', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ to, subject, text })
        });
    } catch (err) {
        console.error('Email notification failed', err);
    }
}

/**
 * Send a Telegram notification using the Telegram Bot API. Requires a bot
 * token and chat ID. Errors are logged to the console.
 * @param {string} token  Telegram bot token
 * @param {string} chatId Telegram chat ID
 * @param {string} text   Message text
 */
async function sendTelegramNotification(token, chatId, text) {
    if (!token || !chatId) return;
    try {
        await fetch(`https://api.telegram.org/bot${token}/sendMessage`, {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ chat_id: chatId, text })
        });
    } catch (err) {
        console.error('Telegram notification failed', err);
    }
}

/**
 * Show a browser push notification, if permission is granted.
 * @param {string} title Notification title
 * @param {string} text  Notification body
 */
function sendPushNotification(title, text) {
    if (!('Notification' in window)) return;
    if (Notification.permission === 'granted') {
        new Notification(title, { body: text });
    }
}

/**
 * Notify all configured channels about an important event. Depending on the
 * user's settings, this may send an email, a Telegram message and/or a push
 * notification. Subject line is derived from the type parameter.
 * @param {string} message Message body
 * @param {string} type    One of info|success|warning|error
 */
function notifyAll(message, type = 'info') {
    const emailEl       = document.getElementById('notifyEmail');
    const tgTokenEl     = document.getElementById('telegramToken');
    const tgChatEl      = document.getElementById('telegramChat');
    const emailToggle   = document.getElementById('enableEmail');
    const telegramToggle= document.getElementById('enableTelegram');
    const pushToggle    = document.getElementById('enablePush');
    const emailAddress  = emailEl ? emailEl.value.trim() : '';
    const token         = tgTokenEl ? tgTokenEl.value.trim() : '';
    const chat          = tgChatEl ? tgChatEl.value.trim() : '';
    const emailEnabled  = emailToggle ? emailToggle.checked : false;
    const telegramEnabled= telegramToggle ? telegramToggle.checked : false;
    const pushEnabled   = pushToggle && pushToggle.checked;
    const subject       = `RASTY ${type.toUpperCase()}`;
    // Send email notification only if enabled and an email address is provided
    if (emailEnabled && emailAddress) {
        sendEmailNotification(emailAddress, subject, message);
    }
    // Send Telegram notification only if enabled and both token and chat ID are present
    if (telegramEnabled && token && chat) {
        sendTelegramNotification(token, chat, message);
    }
    // Send push notification if enabled
    if (pushEnabled) {
        sendPushNotification(subject, message);
    }
}

// Request push notification permission when the checkbox is toggled on
document.addEventListener('DOMContentLoaded', () => {
    const pushEl = document.getElementById('enablePush');
    if (pushEl) {
        pushEl.addEventListener('change', async () => {
            if (pushEl.checked) {
                if ('Notification' in window) {
                    try {
                        const result = await Notification.requestPermission();
                        if (result !== 'granted') {
                            pushEl.checked = false;
                            log('Push notification permission denied', 'warning');
                        }
                    } catch (e) {
                        pushEl.checked = false;
                        log('Push notification request failed', 'warning');
                    }
                } else {
                    pushEl.checked = false;
                    log('Push notifications are not supported in this browser', 'warning');
                }
            }
        });
    }
});

// Theme toggle
const body        = document.body;
const themeToggle = document.getElementById('themeToggle');
themeToggle.addEventListener('click', () => {
    const dark = body.classList.toggle('dark');
    localStorage.setItem('rastyTheme', dark ? 'dark' : 'light');
});
// Initialize theme from storage
if (localStorage.getItem('rastyTheme') === 'dark') {
    body.classList.add('dark');
}

// Logout button (clears cookie and reloads to login page)
const logoutBtn = document.getElementById('logoutBtn');
logoutBtn.addEventListener('click', () => {
    document.cookie = 'rasty_session=;expires=Thu, 01 Jan 1970 00:00:00 UTC;path=/;';
    location.href = 'admin/';
});

// Helper to adjust run/pause input types based on selected mode
function adjustRowInputs(row) {
    // Always set inputs to numeric values now that only normal mode is supported
    const runInput = row.querySelector('.campaign-run');
    const pauseInput = row.querySelector('.campaign-pause');
    if (!runInput || !pauseInput) return;
    runInput.type = 'number';
    runInput.placeholder = 'Run min';
    runInput.title = '';
    runInput.min = '1';
    pauseInput.type = 'number';
    pauseInput.placeholder = 'Pause min';
    pauseInput.title = '';
    pauseInput.min = '1';
}

// Campaign row template creation
function createCampaignRow(idValue = '', runValue = '', pauseValue = '') {
    const row = document.createElement('div');
    row.className = 'row campaign-row';
    // Campaign ID input
    const idInput = document.createElement('input');
    idInput.type  = 'text';
    idInput.placeholder = 'ID';
    idInput.value = idValue;
    idInput.className = 'campaign-id';
    // Run time input
    const runInput = document.createElement('input');
    runInput.value = runValue;
    runInput.className = 'campaign-run';
    // Pause time input
    const pauseInput = document.createElement('input');
    pauseInput.value = pauseValue;
    pauseInput.className = 'campaign-pause';
    // Mode selector: normal or smart variations
    const modeSelect = document.createElement('select');
    modeSelect.className = 'campaign-mode';
    const modes = [
        { value: 'normal',    label: 'Normal' },
        { value: 'smart-run', label: 'Smart Run' },
        { value: 'smart-pause', label: 'Smart Pause' },
        { value: 'smart-both', label: 'Smart Both' }
    ];
    modes.forEach(opt => {
        const o = document.createElement('option');
        o.value = opt.value;
        o.textContent = opt.label;
        modeSelect.appendChild(o);
    });
    // Remove button
    const removeBtn = document.createElement('button');
    removeBtn.textContent = '✖';
    removeBtn.className   = 'button';
    removeBtn.addEventListener('click', () => {
        row.remove();
    });
    // Append all inputs to the row
    row.appendChild(idInput);
    row.appendChild(runInput);
    row.appendChild(pauseInput);
    row.appendChild(modeSelect);

    // Status badge
    const statusBadge = document.createElement('span');
    statusBadge.className = 'status-badge active';
    statusBadge.innerText = 'Active';
    // Toggle button
    const toggleBtn = document.createElement('button');
    toggleBtn.className = 'button toggle-status';
    toggleBtn.innerText = 'Deactivate';
    toggleBtn.onclick = () => {
        const id = idInput.value.trim();
        if (!campaignStates[id]) campaignStates[id] = { active: true };
        const currentlyActive = campaignStates[id].active;
        campaignStates[id].active = !currentlyActive;
        updateCampaignRowUI(id);
        // Remove campaign from the scheduler when toggled off
        if (currentlyActive && !campaignStates[id].active) {
            // clear existing timers and stop if running
            if (schedulerTimers[id]) {
                if (schedulerTimers[id].runTimer) clearTimeout(schedulerTimers[id].runTimer);
                if (schedulerTimers[id].pauseTimer) clearTimeout(schedulerTimers[id].pauseTimer);
                delete schedulerTimers[id];
            }
        } else if (!currentlyActive && campaignStates[id].active) {
            // Reactivate: schedule this campaign if the scheduler is running
            if (schedulerRunning) {
                const token = document.getElementById('apiTokenInput').value.trim();
                if (token) {
                    const campaigns = getCampaignData();
                    const campaign   = campaigns.find(c => c.id === id);
                    if (campaign) {
                        scheduleCampaign(campaign, token, 'normal');
                    }
                }
            }
        }
    };
    row.appendChild(statusBadge);
    row.appendChild(toggleBtn);
    row.appendChild(removeBtn);
    // Adjust input types according to current mode
    // Adjust input types (always numeric now)
    adjustRowInputs(row);
    return row;
}

// Add initial campaign row on page load
const campaignList = document.getElementById('campaignList');
campaignList.appendChild(createCampaignRow());

// Add new campaign row on button click
document.getElementById('addCampaign').addEventListener('click', () => {
    campaignList.appendChild(createCampaignRow());
});

// Interval modes have been removed, so no mode change handler is required.

// Helper to parse all campaign settings
function getCampaignData() {
    const rows = document.querySelectorAll('.campaign-row');
    const campaigns = [];
    rows.forEach(row => {
        const id    = row.querySelector('.campaign-id').value.trim();
        const runVal   = row.querySelector('.campaign-run').value.trim();
        const pauseVal = row.querySelector('.campaign-pause').value.trim();
        const modeSel  = row.querySelector('.campaign-mode');
        if (!id) return;
        const mode = modeSel ? modeSel.value : 'normal';
        let runs = [];
        let pauses = [];
        // Helper to parse comma-separated durations
        const parseList = (str) => str.split(',').map(s => parseInt(s.trim())).filter(n => n > 0);
        if (mode === 'smart-run' || mode === 'smart-both') {
            runs = parseList(runVal);
        } else {
            const r = parseInt(runVal);
            if (r > 0) runs = [r];
        }
        if (mode === 'smart-pause' || mode === 'smart-both') {
            pauses = parseList(pauseVal);
        } else {
            const p = parseInt(pauseVal);
            if (p > 0) pauses = [p];
        }
        // Ensure at least one duration in each list; fallback to 1 minute
        if (runs.length === 0) runs = [1];
        if (pauses.length === 0) pauses = [1];
        campaigns.push({ id, runs, pauses });
    });
    return campaigns;
}

// ---------------------------------------------------------------------------
// API functions to play/stop campaigns
//
// In order to ensure we never start a campaign that is not in a safe state
// (for example drafts, in moderation or rejected campaigns), we need to
// inspect the campaign's current status before starting it. According to the
// PropellerAds workflow a campaign should only be resumed when it is either
// paused or stopped. Starting a campaign that is in any other state will
// trigger moderation and could result in a non‑compliant campaign being
// re‑evaluated. The helper below fetches the status using the same bearer
// token and returns a boolean indicating whether the campaign may be
// started. If the API returns an unexpected response or fails, the
// conservative default is to not start the campaign. See also the README
// for a discussion of possible status values (e.g. draft, moderation,
// ready, active, paused, stopped, declined, archived).

/**
 * Fetch the current status of a campaign.  The PropellerAds SSP API exposes
 * campaign details under the `/adv/campaigns/{id}` endpoint.  The response
 * typically includes a `status` field or a `state` field that describes the
 * campaign's lifecycle.  Other properties like `status_id` may also be
 * returned depending on the API version.  This helper attempts to extract
 * whichever of these fields is present.
 *
 * @param {string|number} id   Campaign ID
 * @param {string} token       API bearer token
 * @returns {Promise<null|any>} The status value or null on failure
 */
async function getCampaignStatus(id, token) {
    try {
        const details = await callApi('GET', `/adv/campaigns/${id}`, token);
        if (!details || typeof details !== 'object') return null;
        // The status may reside under different keys depending on API version.
        if (details.status !== undefined) return details.status;
        if (details.state !== undefined) return details.state;
        if (details.status_id !== undefined) return details.status_id;
        return null;
    } catch (err) {
        // Log as a warning; we will err on the safe side and treat unknown status
        // as not runnable (see isCampaignRunnable).
        log(`Status check failed for campaign ${id}: ${err.message}`, 'warning');
        return null;
    }
}

/**
 * Determine whether a campaign can safely be started based on its current
 * status.  Only campaigns whose status is explicitly "paused" or "stopped"
 * (case insensitive) will return true.  All other statuses—including
 * draft, moderation, ready, active, declined or unknown—will return false.
 *
 * If the status returned from the API is a number, the function
 * conservatively returns false, since we cannot reliably map numeric
 * identifiers to textual statuses without a lookup table.
 *
 * @param {any} status         The status value extracted from the API
 * @returns {boolean}          True if the campaign may be started
 */

function isCampaignRunnableStatus(statusCode) {
    // Campaigns with these numeric statuses must NOT be started.
    // 1 = Draft, 2 = Moderation pending, 3 = Rejected, 9 = Completed
    const blocked = [1, 2, 3, 9];
    // If statusCode is not a number (e.g. null or a string) then err on
    // the side of not running.
    if (typeof statusCode !== 'number') {
        return false;
    }
    return !blocked.includes(statusCode);
}

// Keep track of the last status we notified for each campaign. This prevents
// sending duplicate notifications every time the scheduler checks status.
const lastNotifiedStatus = new Map();
async function callApi(method, path, token, body = null) {
    const url = `https://ssp-api.propellerads.com/v5${path}`;
    try {
        const res = await fetch(url, {
            method: method,
            headers: {
                'Content-Type': 'application/json',
                'Authorization': `Bearer ${token}`
            },
            body: body ? JSON.stringify(body) : undefined
        });
        // Rate limit handling
        if (res.status === 429) {
            const retry = res.headers.get('Retry-After');
            const wait  = retry ? parseInt(retry) : 60;
            log(`Rate limit hit, waiting ${wait} seconds`, 'warning');
            await new Promise(resolve => setTimeout(resolve, wait * 1000));
            return callApi(method, path, token, body);
        }
        if (!res.ok) {
            throw new Error(`HTTP ${res.status}`);
        }
        return await res.json();
    } catch (err) {
        log(`API error: ${err.message}`, 'error');
        throw err;
    }
}

// Keep track of rejected campaigns to ensure they are not started or stopped again.
const rejectedCampaigns = new Set();

/**
 * Attempt to start a campaign. If the campaign fails the compliance check
 * it will be marked as rejected and future attempts to start/stop will be
 * skipped. A boolean return indicates whether the campaign actually started.
 *
 * @param {string|number} id   Campaign ID
 * @param {string} token       API bearer token
 * @returns {Promise<boolean>} True if started, false otherwise
 */
async function playCampaign(id, token) {
    if (campaignStates[id] && campaignStates[id].active === false) {
        log(`Campaign ${id} skipped — marked inactive`, 'error');
        // Preserve existing skip reason or assign a default reason
        if (!campaignStates[id].skipReason) {
            campaignStates[id].skipReason = 'Manually inactive';
        }
        // Cancel any timers to prevent further scheduling
        removeSchedulingForCampaign(id);
        updateCampaignRowUI(id);
        return false;
    }
    const key = String(id);
    // Do not attempt to start a campaign that has already been rejected
    if (rejectedCampaigns.has(key)) {
        log(`Campaign ${id} was previously rejected. Start skipped.`, 'error');
        if (!campaignStates[id]) campaignStates[id] = {};
        campaignStates[id].active = false;
        campaignStates[id].skipReason = 'Rejected';
        updateCampaignRowUI(id);
        return false;
    }
    try {
        // Check the campaign's current status before doing anything. We also
        // handle notifications here: notify only when the campaign is
        // rejected (status 3) or approved/ready (status 4). Duplicate
        // notifications are suppressed by tracking the last notified status.
        const status = await getCampaignStatus(id, token);
        // If status is 3 (Rejected) or 4 (Ready/Approved), send a notification
        // but only if we have not already notified about this status for this
        // campaign. After sending, record the status. When the status moves
        // away from 3 or 4 we clear the recorded status so a future 3 or 4
        // will notify again.
        if (typeof status === 'number') {
            const last = lastNotifiedStatus.get(key);
            // Determine which status events to notify based on user preferences
            const rejectToggleEl   = document.getElementById('notifyRejected');
            const approveToggleEl  = document.getElementById('notifyApproved');
            const completeToggleEl = document.getElementById('notifyCompleted');
            const notifyRejection  = rejectToggleEl ? rejectToggleEl.checked : false;
            const notifyApproval   = approveToggleEl ? approveToggleEl.checked : false;
            const notifyCompleted  = completeToggleEl ? completeToggleEl.checked : false;
            if (status === 3 && last !== 3) {
                if (notifyRejection && typeof notifyAll === 'function') {
                    const msg = `Campaign ${id} has been rejected (status ${status}).`;
                    notifyAll(msg, 'error');
                }
                lastNotifiedStatus.set(key, 3);
            } else if (status === 4 && last !== 4) {
                if (notifyApproval && typeof notifyAll === 'function') {
                    const msg = `Campaign ${id} is approved and ready (status ${status}).`;
                    notifyAll(msg, 'success');
                }
                lastNotifiedStatus.set(key, 4);
            } else if (status === 9 && last !== 9) {
                if (notifyCompleted && typeof notifyAll === 'function') {
                    const msg = `Campaign ${id} has completed (status ${status}).`;
                    // Use success type for completed campaigns
                    notifyAll(msg, 'success');
                }
                lastNotifiedStatus.set(key, 9);
            } else if (![3,4,9].includes(status)) {
                // Clear the record if the status moves away from 3, 4 or 9
                lastNotifiedStatus.delete(key);
            }
        }
        // Only run campaigns if they are not in blocked statuses (1,2,3,9).
        if (!isCampaignRunnableStatus(status)) {
            const statusStr = (status === null || status === undefined) ? 'unknown' : String(status);
            log(`Campaign ${id} skipped — status ${status} (${statusNames[status] || 'Unknown'})`, 'error');
            if (!campaignStates[id]) campaignStates[id] = {};
            campaignStates[id].active = false;
            campaignStates[id].skipReason = statusNames[status] || 'Unknown';
            // Remove scheduling and update UI
            removeSchedulingForCampaign(id);
            updateCampaignRowUI(id);
            return false;
        }
        // Perform a compliance check before attempting to start the campaign.
        const approved = await isCampaignApproved(id, token);
        if (!approved) {
            // Mark campaign as rejected to prevent further attempts.
            rejectedCampaigns.add(key);
            log(`Compliance: campaign ${id} is rejected or pending. Removed from scheduler.`, 'error');
            if (!campaignStates[id]) campaignStates[id] = {};
            campaignStates[id].active = false;
            campaignStates[id].skipReason = 'Rejected or pending';
            // Remove scheduling timers to stop further attempts
            removeSchedulingForCampaign(id);
            updateCampaignRowUI(id);
            return false;
        }
        // Status is acceptable and campaign is approved. Attempt to start it.
        await callApi('PUT', '/adv/campaigns/play', token, { campaign_ids: [parseInt(id)] });
        log(`Started campaign ${id}`, 'success');
        return true;
    } catch (err) {
        log(`Failed to start campaign ${id}`, 'error');
        if (!campaignStates[id]) campaignStates[id] = {};
        campaignStates[id].active = false;
        campaignStates[id].skipReason = 'Start failed';
        // Cancel any timers for this campaign
        removeSchedulingForCampaign(id);
        updateCampaignRowUI(id);
        return false;
    }
}

/**
 * Stop a campaign. If the campaign has been marked as rejected no API call
 * will be made.
 *
 * @param {string|number} id  Campaign ID
 * @param {string} token      API bearer token
 */
async function stopCampaign(id, token) {
    const key = String(id);
    if (rejectedCampaigns.has(key)) {
        return;
    }
    try {
        await callApi('PUT', '/adv/campaigns/stop', token, { campaign_ids: [parseInt(id)] });
        // Stopped messages are logged as errors to emphasize red colouring
        log(`Stopped campaign ${id}`, 'error');
    } catch (err) {
        log(`Failed to stop campaign ${id}`, 'error');
    }
}

// Scheduler state variables
let schedulerRunning  = false;
let schedulerTimers   = {};
let globalCycleActive = false;
let globalCycleTimer1 = null;
let globalCycleTimer2 = null;

/**
 * Remove all scheduled timers for a given campaign and delete its entry
 * from the schedulerTimers object. This helper is called whenever a
 * campaign is marked inactive due to skip, rejection or API failure to
 * ensure it will not continue cycling through run/pause intervals.
 *
 * @param {string|number} id The campaign identifier
 */
function removeSchedulingForCampaign(id) {
    if (schedulerTimers[id]) {
        const timers = schedulerTimers[id];
        if (timers.runTimer) clearTimeout(timers.runTimer);
        if (timers.pauseTimer) clearTimeout(timers.pauseTimer);
        delete schedulerTimers[id];
    }
}

// Main scheduling loop per campaign
// Accepts arrays of run and pause durations to allow rotating through values for smart/advanced modes
function scheduleCampaign(campaign, token, mode) {
    const { id, runs, pauses } = campaign;
    // Initialise schedulerTimers entry if not present
    if (!schedulerTimers[id]) {
        schedulerTimers[id] = { runTimer: null, pauseTimer: null };
    }
    let runIndex = 0;
    let pauseIndex = 0;
    // Recursive cycle function
    const cycle = () => {
        if (!schedulerRunning) return;
        // If the campaign was flagged as rejected, skip any further scheduling
        if (rejectedCampaigns.has(String(id))) {
            log(`Scheduler: campaign ${id} is rejected and will no longer be scheduled.`, 'error');
            return;
        }
        // Skip scheduling if this campaign has been marked inactive
        if (campaignStates[id] && campaignStates[id].active === false) {
            return;
        }
        const runMin = runs[runIndex % runs.length];
        const pauseMin = pauses[pauseIndex % pauses.length];
        // Start the campaign
        playCampaign(id, token);
        // After run duration, stop campaign
        schedulerTimers[id].runTimer = setTimeout(async () => {
            await stopCampaign(id, token);
            if (!schedulerRunning) return;
            // After pause duration, start next cycle
            schedulerTimers[id].pauseTimer = setTimeout(() => {
                if (!schedulerRunning) return;
                // Increment both indices to rotate through lists
                runIndex = (runIndex + 1) % runs.length;
                pauseIndex = (pauseIndex + 1) % pauses.length;
                cycle();
            }, pauseMin * 60 * 1000);
        }, runMin * 60 * 1000);
    };
    cycle();
}

function startScheduler() {
    if (schedulerRunning) {
        log('Scheduler already running', 'warning');
        return;
    }
    const token = document.getElementById('apiTokenInput').value.trim();
    if (!token) {
        log('API token is required', 'warning');
        return;
    }
    const campaigns = getCampaignData();
    if (campaigns.length === 0) {
        log('Please add at least one campaign', 'warning');
        return;
    }
    // Only normal mode is supported; start the scheduler
    const mode = 'normal';
    schedulerRunning = true;
    log('Scheduler started', 'info');
    // Schedule each campaign
    campaigns.forEach(c => {
        scheduleCampaign(c, token, mode);
    });
    // Immediately fetch statistics to populate stats and mark API status
    fetchStatistics(token);
    // Assume API is active once the scheduler starts with a token
    updateApiStatus(true);
}

function stopScheduler() {
    if (!schedulerRunning) {
        log('Scheduler not running', 'warning');
        return;
    }
    schedulerRunning = false;
    // Clear all timers for each campaign
    Object.keys(schedulerTimers).forEach(id => {
        const timers = schedulerTimers[id];
        if (!timers) return;
        if (timers.runTimer) clearTimeout(timers.runTimer);
        if (timers.pauseTimer) clearTimeout(timers.pauseTimer);
    });
    schedulerTimers = {};
    log('Scheduler stopped', 'warning');
    // Stop all running campaigns
    const token = document.getElementById('apiTokenInput').value.trim();
    const campaigns = getCampaignData();
    campaigns.forEach(c => stopCampaign(c.id, token));
}
// Global cycle functionality has been removed

// Progress bar logic
let progressInterval;
let progressSeconds;
const bar = document.getElementById('progressBar');
const txt = document.getElementById('progressText');
function startProgress(mode, totalSeconds) {
    stopProgress();
    progressSeconds = totalSeconds;
    updateProgress(mode);
    progressInterval = setInterval(() => {
        progressSeconds--;
        updateProgress(mode);
        if (progressSeconds <= 0) {
            clearInterval(progressInterval);
        }
    }, 1000);
}
function stopProgress() {
    clearInterval(progressInterval);
    bar.style.width = '0';
    txt.textContent = '';
}
function updateProgress(mode) {
    const total = progressSeconds > 0 ? progressSeconds + 1 : progressSeconds; // avoid negative
    let width = (total > 0) ? ((progressSeconds / total) * 100) : 0;
    bar.style.width = `${100 - width}%`;
    const m = Math.floor(progressSeconds / 60);
    const s = ('0' + (progressSeconds % 60)).slice(-2);
    txt.textContent = `${mode} — ${m}:${s} remaining`;
}// Clear Logs button
const clearBtn = document.getElementById('clearLog');
if (clearBtn) {
    clearBtn.addEventListener('click', () => {
        logBox.innerHTML = '';
        log('Logs cleared.', 'warning');
    });
}
// Display local (browser) time without relying on server connectivity. This
// function updates the clock every second and does not perform any network
// requests, ensuring offline operation. The time is formatted according to
// the user's locale.
const serverTimeEl = document.getElementById('serverTime');
if (serverTimeEl) {
    function updateLocalTime() {
        const now  = new Date();
        const time = now.toLocaleTimeString();
        serverTimeEl.textContent = '🕒 ' + time;
    }
    updateLocalTime();
    setInterval(updateLocalTime, 1000);
}
// Attach Start/Stop buttons to scheduler functions with working hours and timer support
const startBtn = document.getElementById('startScheduler');
const stopBtn = document.getElementById('stopScheduler');
if (startBtn) {
    startBtn.addEventListener('click', () => {
        // Mark that the user wants the scheduler to run
        manualSchedulerStarted = true;
        // Immediately enforce working hours and timer logic
        checkWorkingHoursAndTimer();
    });
}
if (stopBtn) {
    stopBtn.addEventListener('click', () => {
        // User explicitly stops scheduler; disable auto-restart
        manualSchedulerStarted = false;
        // Clear timer if running
        if (timerIntervalHandle) {
            clearInterval(timerIntervalHandle);
            timerIntervalHandle = null;
        }
        timerEndTimestamp = null;
        updateTimerCountdown();
        stopScheduler();
    });
}

function updateCampaignRowUI(id){
    const rows=document.querySelectorAll('.campaign-row');
    rows.forEach(r=>{
        const idInput=r.querySelector('.campaign-id');
        if(idInput && idInput.value.trim()==id){
            const badge=r.querySelector('.status-badge');
            const btn=r.querySelector('.toggle-status');
            if(campaignStates[id].active){
                badge.className='status-badge active';
                badge.innerText='Active';
                btn.innerText='Deactivate';
                r.style.opacity='1';
            } else {
                badge.className='status-badge inactive';
                badge.innerText='Inactive';
                btn.innerText='Activate';
                r.style.opacity='0.5';
            }
        }
    });
}

// -------------------------- Working hours & Timer additions --------------------------

// Maintain a 7x24 schedule matrix. Index 0 is Monday, 6 is Sunday. Each element is a boolean indicating
// whether the scheduler is allowed to run at that hour. By default all hours are enabled.
let workingSchedule = [];
for (let d = 0; d < 7; d++) {
    const hours = [];
    for (let h = 0; h < 24; h++) {
        hours.push(true);
    }
    workingSchedule.push(hours);
}

// When the user clicks the start button we set this flag so that the scheduler will automatically
// resume when entering working hours. When false, automatic start/resume is disabled.
let manualSchedulerStarted = false;

// Timer end timestamp (in ms) when the scheduler should automatically stop. Null when no timer is set.
let timerEndTimestamp = null;
let timerIntervalHandle = null;

// Track mouse dragging state for multi-hour selection on the working hours grid
let isDragging = false;
let dragSetToActive = false;

// User toggles to enable or disable working hours and timer functionality. Default to false (disabled).
let enableWorkingHours = false;
let enableTimerOption = false;

/**
 * Build the working hours grid table and attach click handlers for toggling individual cells.
 */
function buildWorkingHoursTable() {
    const table = document.getElementById('workingHoursTable');
    if (!table) return;
    // Clear any existing content
    table.innerHTML = '';
    // Header row
    const headerRow = document.createElement('tr');
    const emptyTh = document.createElement('th');
    emptyTh.textContent = '';
    headerRow.appendChild(emptyTh);
    for (let h = 0; h < 24; h++) {
        const th = document.createElement('th');
        th.textContent = h.toString().padStart(2, '0');
        headerRow.appendChild(th);
    }
    table.appendChild(headerRow);
    // Day names for Monday-Sunday
    const dayNames = ['Mo', 'Tu', 'We', 'Th', 'Fr', 'Sa', 'Su'];
    for (let d = 0; d < 7; d++) {
        const row = document.createElement('tr');
        const dayCell = document.createElement('th');
        dayCell.textContent = dayNames[d];
        row.appendChild(dayCell);
        for (let h = 0; h < 24; h++) {
            const cell = document.createElement('td');
            cell.dataset.day = d;
            cell.dataset.hour = h;
            cell.classList.add(workingSchedule[d][h] ? 'active' : 'inactive');
            cell.textContent = '';
            // Remove simple click handler and instead set up drag selection handlers
            cell.addEventListener('mousedown', (e) => {
                isDragging = true;
                const day = parseInt(cell.dataset.day);
                const hour = parseInt(cell.dataset.hour);
                // Determine whether to enable or disable based on current state
                dragSetToActive = !workingSchedule[day][hour];
                toggleCellState(cell, day, hour, dragSetToActive);
                // Prevent text selection while dragging
                e.preventDefault();
            });
            cell.addEventListener('mouseover', () => {
                if (!isDragging) return;
                const day = parseInt(cell.dataset.day);
                const hour = parseInt(cell.dataset.hour);
                toggleCellState(cell, day, hour, dragSetToActive);
            });
            row.appendChild(cell);
        }
        table.appendChild(row);
    }
    // End dragging when mouse button is released anywhere on the document
    document.addEventListener('mouseup', () => {
        isDragging = false;
    });
}

/**
 * Update all cells' classes to reflect the current workingSchedule state. Useful after bulk
 * operations such as selecting all working days or clearing all.
 */
function refreshWorkingHoursTable() {
    const table = document.getElementById('workingHoursTable');
    if (!table) return;
    const cells = table.querySelectorAll('td');
    cells.forEach(cell => {
        const day = parseInt(cell.dataset.day);
        const hour = parseInt(cell.dataset.hour);
        const active = workingSchedule[day][hour];
        if (active) {
            cell.classList.add('active');
            cell.classList.remove('inactive');
        } else {
            cell.classList.add('inactive');
            cell.classList.remove('active');
        }
    });
}

/**
 * Toggle a single schedule cell to a specified state and update its classes. If the cell is
 * already in the desired state, no action is taken.
 * @param {HTMLElement} cell  The table cell element
 * @param {number} day        Day index (0=Mon..6=Sun)
 * @param {number} hour       Hour index (0-23)
 * @param {boolean} newState  True to enable, false to disable
 */
function toggleCellState(cell, day, hour, newState) {
    if (workingSchedule[day][hour] === newState) return;
    workingSchedule[day][hour] = newState;
    if (newState) {
        cell.classList.add('active');
        cell.classList.remove('inactive');
    } else {
        cell.classList.add('inactive');
        cell.classList.remove('active');
    }
}

/**
 * Determine whether the current time falls within the user-defined working hours schedule.
 * Uses the browser's local time. Returns true if the scheduler should run now.
 *
 * @returns {boolean} True if within working hours
 */
function isInWorkingHours() {
    // If the schedule table is not present we assume no restriction (always true)
    const table = document.getElementById('workingHoursTable');
    if (!table) return true;
    const now = new Date();
    // getDay: 0 (Sunday) to 6 (Saturday). Convert to schedule index (0=Monday, 6=Sunday)
    let dayIndex = now.getDay();
    // Map to Monday-first indexing
    dayIndex = (dayIndex + 6) % 7;
    const hour = now.getHours();
    // Safety: if schedule matrix is missing, treat as always allowed
    if (!workingSchedule[dayIndex]) return true;
    return workingSchedule[dayIndex][hour];
}

/**
 * Called periodically to enforce working hours and timer restrictions. If the manualSchedulerStarted flag
 * is true, the scheduler will automatically start when entering a working period and stop when
 * leaving one. If a timer is active, the scheduler will be stopped when it expires.
 */
function checkWorkingHoursAndTimer() {
    // Timer check: only enforce when the timer option is enabled
    if (enableTimerOption && timerEndTimestamp) {
        const nowMs = Date.now();
        if (nowMs >= timerEndTimestamp) {
            timerEndTimestamp = null;
            updateTimerCountdown();
            if (schedulerRunning) {
                stopScheduler();
                log('Scheduler stopped: timer expired', 'warning');
            }
        }
    }
    // If the user requested the scheduler to run
    if (manualSchedulerStarted) {
        // Determine whether current time is allowed based on working hours setting
        const allowed = enableWorkingHours ? isInWorkingHours() : true;
        if (allowed) {
            // If scheduler is not running, attempt to start it
            if (!schedulerRunning) {
                // Only start if campaigns and token are available
                const tokenEl = document.getElementById('apiTokenInput');
                const token = tokenEl ? tokenEl.value.trim() : '';
                const campaigns = (typeof getCampaignData === 'function') ? getCampaignData() : [];
                if (token && campaigns.length > 0) {
                    startScheduler();
                }
            }
        } else {
            // Not allowed, stop if running
            if (schedulerRunning) {
                stopScheduler();
                log('Scheduler stopped: outside working hours', 'warning');
            }
        }
    }
}

/**
 * Update the timer countdown display element. Called when timer is set and every second
 * thereafter. When no timer is active, clears the display.
 */
function updateTimerCountdown() {
    const timerEl = document.getElementById('timerCountdown');
    if (!timerEl) return;
    if (!timerEndTimestamp) {
        timerEl.textContent = '';
        return;
    }
    const remaining = timerEndTimestamp - Date.now();
    if (remaining <= 0) {
        timerEl.textContent = '';
        timerEndTimestamp = null;
        return;
    }
    const m = Math.floor(remaining / 60000);
    const s = Math.floor((remaining % 60000) / 1000);
    timerEl.textContent = `${m}:${s.toString().padStart(2, '0')} remaining`;
}

/**
 * Handler for the Set Timer button. Reads the number of minutes and sets the timer accordingly.
 */
function setTimer() {
    const input = document.getElementById('timerMinutes');
    if (!input) return;
    const minutes = parseInt(input.value, 10);
    if (isNaN(minutes) || minutes <= 0) {
        log('Please enter a positive number of minutes for the timer.', 'warning');
        return;
    }
    timerEndTimestamp = Date.now() + minutes * 60 * 1000;
    log(`Scheduler timer set for ${minutes} minute${minutes !== 1 ? 's' : ''}.`, 'info');
    updateTimerCountdown();
    // Clear any existing interval
    if (timerIntervalHandle) clearInterval(timerIntervalHandle);
    // Update countdown every second
    timerIntervalHandle = setInterval(updateTimerCountdown, 1000);
}

/**
 * Register click handlers for the schedule control buttons (All, Working Days, Weekend, Clear All).
 */
function initScheduleControls() {
    const container = document.getElementById('scheduleControls');
    if (!container) return;
    container.querySelectorAll('.schedule-btn').forEach(btn => {
        btn.addEventListener('click', () => {
            const action = btn.dataset.action;
            if (action === 'all') {
                for (let d = 0; d < 7; d++) {
                    for (let h = 0; h < 24; h++) {
                        workingSchedule[d][h] = true;
                    }
                }
            } else if (action === 'working') {
                // Monday (0) to Friday (4) active, weekend cleared
                for (let d = 0; d < 7; d++) {
                    for (let h = 0; h < 24; h++) {
                        workingSchedule[d][h] = (d >= 0 && d <= 4);
                    }
                }
            } else if (action === 'weekend') {
                // Saturday (5) and Sunday (6) active, others cleared
                for (let d = 0; d < 7; d++) {
                    for (let h = 0; h < 24; h++) {
                        workingSchedule[d][h] = (d >= 5);
                    }
                }
            } else if (action === 'clear') {
                for (let d = 0; d < 7; d++) {
                    for (let h = 0; h < 24; h++) {
                        workingSchedule[d][h] = false;
                    }
                }
            }
            refreshWorkingHoursTable();
        });
    });
}

/**
 * Initialise working hours grid, schedule controls and timer controls. Should be called when
 * the DOM is fully loaded. Also sets up periodic checks for working hours and timer.
 */
function initWorkingHoursAndTimer() {
    buildWorkingHoursTable();
    initScheduleControls();
    // Attach timer button handler
    const timerBtn = document.getElementById('setTimer');
    if (timerBtn) {
        timerBtn.addEventListener('click', setTimer);
    }
    // Initialise working hours and timer enable checkboxes
    const whChk = document.getElementById('enableWorkingHours');
    if (whChk) {
        enableWorkingHours = whChk.checked;
        whChk.addEventListener('change', (e) => {
            enableWorkingHours = e.target.checked;
        });
    }
    const tmChk = document.getElementById('enableTimerOption');
    if (tmChk) {
        enableTimerOption = tmChk.checked;
        tmChk.addEventListener('change', (e) => {
            enableTimerOption = e.target.checked;
        });
    }
    // Start periodic check every 30 seconds to enforce working hours and timer
    setInterval(checkWorkingHoursAndTimer, 30000);
    // Also update the countdown display every second if a timer is set (timerIntervalHandle handles this)
}

async function fetchStatistics(token){
    try{
        const today = new Date().toISOString().slice(0,10);
        const url   = `https://ssp-api.propellerads.com/v5/adv/statistics?group_by=day&date_from=${today}&date_to=${today}`;
        const res   = await fetch(url, {
            headers: { 'Authorization': `Bearer ${token}` }
        });
        // If the response status is in the 2xx or 4xx range treat the API as reachable
        if(res.status >= 200 && res.status < 500) {
            updateApiStatus(true);
        } else {
            updateApiStatus(false);
        }
        // Only attempt to parse JSON and update stats if response is OK
        if(res.ok) {
            const data = await res.json();
            const row  = (data && data.data && data.data[0]) ? data.data[0] : null;
            if(row){
                const impressions = row.impressions || 0;
                const spent       = row.cost || 0;
                const traffic     = impressions; // raw number as requested
                document.getElementById("statImpressions").innerText = impressions.toLocaleString();
                document.getElementById("statSpent").innerText       = "$" + spent.toFixed(2);
                document.getElementById("statTraffic").innerText     = traffic.toLocaleString();
            }
        }
    } catch(e) {
        // On any error mark the API as inactive
        updateApiStatus(false);
    }
}

setInterval(() => {
    const token = document.getElementById("apiTokenInput")?.value;
    if(token){
        // When a token is present attempt to fetch statistics. The call will
        // update the API status indicator on success or failure.
        fetchStatistics(token);
    } else {
        // No token – mark API inactive
        updateApiStatus(false);
    }
// Fetch statistics every hour (3600000 ms) instead of every minute
}, 3600000);

// ---------------------------------------------------------
// Additional utility functions
// Download logs as a plain text file. Creates a temporary blob and
// triggers an automatic download.
function downloadLogs() {
    const logText = logBox.innerText || '';
    const blob    = new Blob([logText], { type: 'text/plain' });
    const url     = URL.createObjectURL(blob);
    const anchor  = document.createElement('a');
    anchor.href = url;
    anchor.download = `mediamoon_logs_${Date.now()}.txt`;
    anchor.style.display = 'none';
    document.body.appendChild(anchor);
    anchor.click();
    document.body.removeChild(anchor);
    URL.revokeObjectURL(url);
}

// Remove all campaign rows and clear internal state
function resetCampaigns() {
    const list = document.getElementById('campaignList');
    if (list) list.innerHTML = '';
    for (const key in campaignStates) {
        if (Object.prototype.hasOwnProperty.call(campaignStates, key)) {
            delete campaignStates[key];
        }
    }
    log('All campaigns removed.', 'warning');
}

// Attach additional buttons to their handlers
const dlBtn = document.getElementById('downloadLogs');
if (dlBtn) dlBtn.addEventListener('click', downloadLogs);
const resetBtn = document.getElementById('resetCampaigns');
if (resetBtn) resetBtn.addEventListener('click', resetCampaigns);

// Initialise working hours grid, schedule controls and timer when page loads.
try {
    initWorkingHoursAndTimer();
} catch (e) {
    // Ignore initialisation errors if the elements are not present (e.g., on login page)
}
